<?php
// update_order.php
// Returns JSON only

// capture accidental output so we always return valid JSON
ob_start();

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', 0); // don't show errors in response
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . "/php_errors.log");

// make mysqli throw exceptions for easier handling
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$response = ['status' => 'error', 'message' => 'Unknown error'];

try {
    // --- DB connect ---
    // $host = "localhost";
    // $user = "root";
    // $pass = "";
    // $db   = "restaurant_pos";

//     // 🔹 DB connect for live server 
// $host = "localhost";
// $user = "phizixcl_pos_user";
// $pass = "ta2024@sgh.t/?!@";
// $db   = "phizixcl_restaurant_pos";

include "db.php";


    $conn = new mysqli($host, $user, $pass, $db);
    $conn->set_charset('utf8mb4');

    // --- Read input (JSON preferred) ---
    $input = file_get_contents('php://input');
    $data  = json_decode($input, true);
    if (!$data) $data = $_POST;

    if (empty($data)) {
        throw new Exception("No input data");
    }

    $table = intval($data['table'] ?? 0);
    $type  = $data['type'] ?? 'dinein';
    $cart  = $data['cart'] ?? [];

    if ($table <= 0) throw new Exception("Invalid table");
    if (!is_array($cart) || count($cart) === 0) throw new Exception("Cart is empty");

    // --- Find latest active order for this table ---
    $sql = "SELECT id FROM orders 
            WHERE table_no=? AND order_type=? AND order_status IN ('pending','booked')
            ORDER BY id DESC LIMIT 1";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("is", $table, $type);
    $stmt->execute();
    $res = $stmt->get_result();
    $order = $res->fetch_assoc();
    $stmt->close();

    if (!$order) {
        throw new Exception("No active order found for this table");
    }

    $order_id = intval($order['id']);

    // --- Start transaction ---
    $conn->begin_transaction();

    // 1) Delete existing order_items for this order
    $del = $conn->prepare("DELETE FROM order_items WHERE order_id = ?");
    $del->bind_param("i", $order_id);
    $del->execute();
    $del->close();

    // 2) Insert new order_items
    $ins = $conn->prepare("
        INSERT INTO order_items (order_id, product_id, product_name, qty, price, total)
        VALUES (?, ?, ?, ?, ?, ?)
    ");

    $grandTotal = 0.0;
    foreach ($cart as $item) {
        $pid   = intval($item['id'] ?? 0);
        $pname = (string)($item['name'] ?? '');
        $qty   = max(1, intval($item['qty'] ?? 1));
        $price = floatval($item['price'] ?? 0);
        $total = $qty * $price;

        $ins->bind_param("iisidd", $order_id, $pid, $pname, $qty, $price, $total);
        $ins->execute();

        $grandTotal += $total;
    }
    $ins->close();

    // 3) Update final_total in orders (use final_total column)
    //    (do NOT update non-existing columns like 'total' or 'updated_at' unless they exist)
    $upd = $conn->prepare("UPDATE orders SET final_total = ? WHERE id = ?");
    $upd->bind_param("di", $grandTotal, $order_id);
    $upd->execute();
    $upd->close();

    $conn->commit();

    $response = [
        "status" => "success",
        "message" => "Order updated successfully",
        "order_id" => $order_id,
        "receipt_url" => "receipt_pdf.php?order_id={$order_id}&print=1"
    ];
} catch (Throwable $e) {
    // rollback if we started a transaction
    if (isset($conn) && $conn instanceof mysqli && $conn->errno === 0) {
        @$conn->rollback();
    }
    // log the actual exception message to php_errors.log (already configured)
    $response = [
        "status" => "error",
        "message" => $e->getMessage()
    ];
}

// capture accidental output (if any) and include it in debug_output (safe)
$extra = ob_get_clean();
if ($extra !== '') $response['debug_output'] = $extra;

// final JSON
echo json_encode($response, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
exit;
