<?php
include "db.php";
header('Content-Type: application/json');

if ($conn->connect_error) {
    echo json_encode(["success" => false, "message" => "Database connection failed"]);
    exit;
}

error_reporting(E_ALL);
ini_set('display_errors', 1);

// ✅ Get JSON input
$data = json_decode(file_get_contents("php://input"), true);
if (!$data) {
    echo json_encode(["success" => false, "message" => "No data received"]);
    exit;
}

// ✅ Extract data
$supplier_id     = intval($data['supplier_id']);
$paid            = floatval($data['paid']);
$unpaid          = floatval($data['unpaid']);
$payment_method  = strtolower(trim($data['payment_method']));
$cart            = $data['cart'];

// ✅ Calculate total
$cartTotal = 0.0;
foreach ($cart as $item) {
    $cartTotal += floatval($item['total']);
}

try {
    $conn->begin_transaction();

    // ✅ 1️⃣ Insert into purchases
    $stmt = $conn->prepare("
        INSERT INTO purchases (supplier_id, subtotal, paid, unpaid, payment_method)
        VALUES (?, ?, ?, ?, ?)
    ");
    if (!$stmt) throw new Exception("Prepare failed (purchases): " . $conn->error);
    $stmt->bind_param("iddds", $supplier_id, $cartTotal, $paid, $unpaid, $payment_method);
    $stmt->execute();
    $purchase_id = $stmt->insert_id;
    $stmt->close();

    if ($purchase_id <= 0) throw new Exception("Purchase insert failed");

    // ✅ 2️⃣ Insert purchase items
    $itemStmt = $conn->prepare("
        INSERT INTO purchase_items (purchase_id, product_name, price, qty, qty_type, discount, total)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    if (!$itemStmt) throw new Exception("Prepare failed (purchase_items): " . $conn->error);

    foreach ($cart as $item) {
        $product   = trim($item['product']);
        $price     = floatval($item['price']);
        $qty       = floatval($item['qty']);
        $qty_type  = trim($item['qty_type']);
        $discount  = floatval($item['discount']);
        $total     = floatval($item['total']);

        $itemStmt->bind_param("isdssdd", $purchase_id, $product, $price, $qty, $qty_type, $discount, $total);
        if (!$itemStmt->execute()) throw new Exception("Insert failed (purchase_items): " . $itemStmt->error);

        // ✅ 3️⃣ Fetch product type from pu_product
        $typeResult = $conn->prepare("SELECT product_type FROM pu_product WHERE p_product_name = ?");
        $typeResult->bind_param("s", $product);
        $typeResult->execute();
        $typeRow = $typeResult->get_result()->fetch_assoc();
        $typeResult->close();

        if ($typeRow) {
            $type = trim($typeRow['product_type']);

            // ✅ Manage stock according to type
            if ($type === 'Manage Fridge') {
                // 🔹 Fridge stock update
                $check = $conn->prepare("SELECT qty, qty_type FROM current_stock WHERE product_name = ?");
                $check->bind_param("s", $product);
                $check->execute();
                $result = $check->get_result();

                if ($result->num_rows > 0) {
                    // Update qty, keep existing qty_type if none provided
                    $row = $result->fetch_assoc();
                    $existingQtyType = $row['qty_type'] ?? $qty_type;
                    $update = $conn->prepare("
                        UPDATE current_stock 
                        SET qty = qty + ?, qty_type = ?, updated_at = NOW() 
                        WHERE product_name = ?
                    ");
                    $update->bind_param("dss", $qty, $existingQtyType, $product);
                    $update->execute();
                    $update->close();
                } else {
                    // Insert new record
                    $add = $conn->prepare("
                        INSERT INTO current_stock (product_name, qty, qty_type, updated_at) 
                        VALUES (?, ?, ?, NOW())
                    ");
                    $add->bind_param("sds", $product, $qty, $qty_type);
                    $add->execute();
                    $add->close();
                }
                $check->close();

            } elseif ($type === 'Solid Product') {
                // 🔹 Solid stock update
                $check = $conn->prepare("SELECT qty, qty_type FROM solid_stock WHERE product_name = ?");
                $check->bind_param("s", $product);
                $check->execute();
                $result = $check->get_result();

                if ($result->num_rows > 0) {
                    // Update qty, keep existing qty_type if none provided
                    $row = $result->fetch_assoc();
                    $existingQtyType = $row['qty_type'] ?? $qty_type;
                    $update = $conn->prepare("
                        UPDATE solid_stock 
                        SET qty = qty + ?, qty_type = ?, updated_at = NOW() 
                        WHERE product_name = ?
                    ");
                    $update->bind_param("dss", $qty, $existingQtyType, $product);
                    $update->execute();
                    $update->close();
                } else {
                    // Insert new record
                    $add = $conn->prepare("
                        INSERT INTO solid_stock (product_name, qty, qty_type, updated_at) 
                        VALUES (?, ?, ?, NOW())
                    ");
                    $add->bind_param("sds", $product, $qty, $qty_type);
                    $add->execute();
                    $add->close();
                }
                $check->close();
            }
            // ✅ If type is "Unmanaged Stock" — skip stock update
        }
    }
    $itemStmt->close();

    // ✅ 4️⃣ Update supplier balance
    $conn->query("UPDATE suppliers SET balance = balance + $unpaid WHERE id = $supplier_id");

    // ✅ 5️⃣ Record transaction
    $typeForTx = (in_array($payment_method, ['bank', 'card'])) ? 'bank' : 'cash';
    $direction   = 'out';
    $category    = 'purchase';
    $description = "Purchase #{$purchase_id} to Supplier #{$supplier_id}";

    // Get previous balance
    $prevBal = 0.00;
    $balStmt = $conn->prepare("SELECT current_balance FROM transactions WHERE type=? ORDER BY id DESC LIMIT 1");
    if ($balStmt) {
        $balStmt->bind_param("s", $typeForTx);
        $balStmt->execute();
        $res = $balStmt->get_result();
        if ($res && $res->num_rows > 0) {
            $prevBal = floatval($res->fetch_assoc()['current_balance']);
        }
        $balStmt->close();
    }

    $newBal = $prevBal - $paid;

    $tx = $conn->prepare("
        INSERT INTO transactions (type, category, description, amount, previous_balance, current_balance, direction, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    if (!$tx) throw new Exception("Prepare failed (transactions): " . $conn->error);

    $tx->bind_param("sssddds", $typeForTx, $category, $description, $paid, $prevBal, $newBal, $direction);
    $tx->execute();
    $tx->close();

    // ✅ 6️⃣ Update cash/bank balance
    $balTable = ($typeForTx === 'cash') ? "cash_balance" : "bank_balance";
    $check = $conn->query("SELECT id FROM $balTable WHERE id=1");
    if ($check->num_rows == 0) {
        $conn->query("INSERT INTO $balTable (id, current_balance) VALUES (1, 0)");
    }

    $updateBal = $conn->prepare("UPDATE $balTable SET current_balance = ? WHERE id = 1");
    $updateBal->bind_param("d", $newBal);
    $updateBal->execute();
    $updateBal->close();

    $conn->commit();

    echo json_encode([
        "success" => true,
        "message" => "✅ Purchase saved successfully and stock updated!",
        "purchase_id" => $purchase_id,
        "cart_total" => $cartTotal
    ]);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(["success" => false, "message" => "❌ Error: " . $e->getMessage()]);
}

$conn->close();
?>
