<?php
include "db.php";
header("Content-Type: application/json");
session_start();

// Read JSON input
$data = json_decode(file_get_contents("php://input"), true);
if (!$data || !isset($data['cart']) || count($data['cart']) == 0) {
    echo json_encode(["success" => false, "message" => "Invalid data"]);
    exit;
}

// Extract order details
$order_id       = intval($data['order_id'] ?? 0);
$customer_name  = $data['name'] ?? 'Guest';
$customer_phone = $data['phone'] ?? '';
$payment_type   = strtolower($data['payment'] ?? 'cash');
$subtotal       = floatval($data['subtotal'] ?? 0);
$discount       = floatval($data['discount'] ?? 0);
$tip            = floatval($data['tip'] ?? 0);
$final_total    = floatval($data['finalTotal'] ?? 0);

if ($order_id <= 0) {
    echo json_encode(["success" => false, "message" => "Missing or invalid order ID"]);
    exit;
}

try {
    $conn->begin_transaction();

    // 1️⃣ Get old order details
    $oldOrderRes = $conn->query("SELECT * FROM orders WHERE id=$order_id");
    if (!$oldOrderRes || $oldOrderRes->num_rows === 0) {
        throw new Exception("Original order not found");
    }
    $oldOrder = $oldOrderRes->fetch_assoc();
    $oldPaymentType = strtolower($oldOrder['payment_type']);
    $oldFinalTotal  = floatval($oldOrder['final_total']);

    // 2️⃣ Get old items
    $oldItemsRes = $conn->query("SELECT * FROM order_items WHERE order_id=$order_id");
    $oldItems = [];
    while ($row = $oldItemsRes->fetch_assoc()) {
        $oldItems[$row['product_name']] = $row;
    }

    // 3️⃣ Reverse old transaction from balance
    $balTableOld = ($oldPaymentType === 'bank') ? "bank_balance" : "cash_balance";
    $resOldBal = $conn->query("SELECT current_balance FROM $balTableOld WHERE id=1");
    $oldBal = ($resOldBal && $resOldBal->num_rows > 0) ? floatval($resOldBal->fetch_assoc()['current_balance']) : 0.00;
    $newOldBal = $oldBal - $oldFinalTotal;
    $conn->query("UPDATE $balTableOld SET current_balance = $newOldBal WHERE id=1");

    // Delete old transaction
    $desc = "Collection Order #{$order_id}";
    $conn->query("DELETE FROM transactions WHERE description = '$desc' AND category='collection_order'");

    // 4️⃣ Update order table
    $stmt = $conn->prepare("UPDATE orders 
        SET customer_name=?, customer_phone=?, payment_type=?, subtotal=?, discount=?, tip=?, final_total=? 
        WHERE id=? AND order_type='collection'");
    $stmt->bind_param("sssddddi", $customer_name, $customer_phone, $payment_type, $subtotal, $discount, $tip, $final_total, $order_id);
    $stmt->execute();
    $stmt->close();

    // 5️⃣ Delete old items
    $conn->query("DELETE FROM order_items WHERE order_id=$order_id");

    // 6️⃣ Insert new items + handle solid stock
    $stmt = $conn->prepare("INSERT INTO order_items (order_id, product_id, product_name, qty, price, total) 
                            VALUES (?, ?, ?, ?, ?, ?)");

    foreach ($data['cart'] as $item) {

        // FIXED: Always get proper product name
        $pid   = intval($item['id'] ?? 0);
        $pname = $item['name'] ?? ($item['product_name'] ?? null);
        $qty   = intval($item['qty'] ?? 0);
        $price = floatval($item['price'] ?? 0);
        $total = $qty * $price;

        if ($pname === null || $pname === "") {
            throw new Exception("Product name missing for product ID: $pid");
        }

        $stmt->bind_param("iisiid", $order_id, $pid, $pname, $qty, $price, $total);
        $stmt->execute();

        // Check if Solid Product
        $chk = $conn->prepare("
            SELECT id FROM pu_product 
            WHERE p_product_name=? AND product_type='Solid Product'
            LIMIT 1
        ");
        $chk->bind_param("s", $pname);
        $chk->execute();
        $res = $chk->get_result();

        if ($res && $res->num_rows > 0) {
            $oldQty = $oldItems[$pname]['qty'] ?? 0;

            if ($qty != $oldQty) {
                $diffQty = $oldQty - $qty;

                $stockQ = $conn->prepare("SELECT id, qty FROM solid_stock WHERE product_name=? LIMIT 1");
                $stockQ->bind_param("s", $pname);
                $stockQ->execute();
                $stockRes = $stockQ->get_result();

                if ($stockRes && $stockRes->num_rows > 0) {
                    $stockRow = $stockRes->fetch_assoc();
                    $newStockQty = max(0, $stockRow['qty'] + $diffQty);

                    $upd = $conn->prepare("UPDATE solid_stock SET qty=?, updated_at=NOW() WHERE id=?");
                    $upd->bind_param("ii", $newStockQty, $stockRow['id']);
                    $upd->execute();
                    $upd->close();

                } else if ($diffQty < 0) {
                    $qtyType = "pcs";
                    $ins = $conn->prepare("INSERT INTO solid_stock (product_name, qty, qty_type, updated_at) 
                                           VALUES (?, ?, ?, NOW())");
                    $ins->bind_param("sis", $pname, abs($diffQty), $qtyType);
                    $ins->execute();
                    $ins->close();
                }

                $stockQ->close();
            }
        }

        $chk->close();
    }

    $stmt->close();

    // 7️⃣ Insert new transaction
    $typeForTx = ($payment_type === 'bank') ? 'bank' : 'cash';
    $direction = 'in';
    $category  = 'collection_order';
    $description = "Collection Order #{$order_id}";

    $resBal = $conn->query("SELECT current_balance FROM {$typeForTx}_balance WHERE id=1");
    $prevBal = ($resBal && $resBal->num_rows > 0) ? floatval($resBal->fetch_assoc()['current_balance']) : 0.00;
    $newBal = $prevBal + $final_total;

    $tx = $conn->prepare("INSERT INTO transactions 
        (type, category, description, amount, previous_balance, current_balance, direction, created_at) 
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW())");
    $tx->bind_param("sssddds", $typeForTx, $category, $description, $final_total, $prevBal, $newBal, $direction);
    $tx->execute();
    $tx->close();

    // 8️⃣ Update balance
    $updateBalance = $conn->prepare("UPDATE {$typeForTx}_balance SET current_balance = ? WHERE id=1");
    $updateBalance->bind_param("d", $newBal);
    $updateBalance->execute();
    $updateBalance->close();

    $conn->commit();

    echo json_encode([
        "success" => true,
        "message" => "Collection order updated successfully with solid stock adjustment ✅",
        "order_id" => $order_id
    ]);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(["success" => false, "message" => "❌ Error: " . $e->getMessage()]);
}

$conn->close();
?>
