<?php
include "db.php";
header("Content-Type: application/json");

// Set timezone
date_default_timezone_set("Asia/Dhaka");
$device_time = date("Y-m-d H:i:s");

// Decode request
$data = json_decode(file_get_contents("php://input"), true);

if (!$data || !isset($data['cart']) || count($data['cart']) == 0) {
    echo json_encode(["success" => false, "message" => "Invalid data"]);
    exit;
}

// Extract fields
$customer_name    = trim($data['name'] ?? 'Guest');
$customer_phone   = trim($data['phone'] ?? '');
$customer_address = trim($data['address'] ?? '');
$payment_type     = strtolower($data['payment'] ?? 'cash');
$subtotal         = floatval($data['subtotal'] ?? 0);
$discount         = floatval($data['discount'] ?? 0);
$tip              = floatval($data['tip'] ?? 0);
$final_total      = floatval($data['finalTotal'] ?? 0);

try {
    $conn->begin_transaction();

    /* STEP 1: Insert / Update Customer */
    $cust_id = null;

    if (!empty($customer_phone)) {
        $stmtCheck = $conn->prepare("SELECT id FROM customers WHERE mobile=?");
        $stmtCheck->bind_param("s", $customer_phone);
        $stmtCheck->execute();
        $res = $stmtCheck->get_result();

        if ($res && $res->num_rows > 0) {
            $row     = $res->fetch_assoc();
            $cust_id = $row['id'];

            $stmtUpd = $conn->prepare("UPDATE customers SET name=?, address=? WHERE id=?");
            $stmtUpd->bind_param("ssi", $customer_name, $customer_address, $cust_id);
            $stmtUpd->execute();
            $stmtUpd->close();
        } else {
            $stmtCust = $conn->prepare("INSERT INTO customers (name, mobile, address, created_at) VALUES (?, ?, ?, ?)");
            $stmtCust->bind_param("ssss", $customer_name, $customer_phone, $customer_address, $device_time);
            $stmtCust->execute();
            $cust_id = $stmtCust->insert_id;
            $stmtCust->close();
        }

        $stmtCheck->close();
    }

    /* STEP 2: Create Order */
    $stmt = $conn->prepare("
        INSERT INTO orders
        (order_type, customer_name, customer_phone, payment_type, subtotal, discount, tip, final_total, created_at)
        VALUES ('collection', ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->bind_param("sssdddds", $customer_name, $customer_phone, $payment_type, $subtotal, $discount, $tip, $final_total, $device_time);
    $stmt->execute();
    $order_id = $stmt->insert_id;
    $stmt->close();

    if ($order_id <= 0) throw new Exception("Order insert failed");

    /* STEP 3: Insert Order Items + Solid Stock Deduction */
    $stmtItem = $conn->prepare("
        INSERT INTO order_items (order_id, product_id, product_name, qty, price, total)
        VALUES (?, ?, ?, ?, ?, ?)
    ");

    foreach ($data['cart'] as $item) {

        $pid   = intval($item['id']);
        $pname = $item['name'];
        $qty   = floatval($item['qty']);
        $price = floatval($item['price']);
        $total = $qty * $price;

        // Insert order item
        $stmtItem->bind_param("iisiid", $order_id, $pid, $pname, $qty, $price, $total);
        if (!$stmtItem->execute()) {
            throw new Exception("Insert failed (order_items): " . $stmtItem->error);
        }

        /* ------------------------------------------
            FIXED SOLID PRODUCT CHECK — BY NAME !!
        -------------------------------------------*/
        $solidCheck = $conn->prepare("
            SELECT id 
            FROM pu_product
            WHERE p_product_name = ?
              AND product_type = 'Solid Product'
            LIMIT 1
        ");
        $solidCheck->bind_param("s", $pname);
        $solidCheck->execute();
        $resSolid = $solidCheck->get_result();

        if ($resSolid && $resSolid->num_rows > 0) {

            /* Fetch solid stock */
            $stockQ = $conn->prepare("
                SELECT qty 
                FROM solid_stock 
                WHERE product_name=? 
                LIMIT 1
            ");
            $stockQ->bind_param("s", $pname);
            $stockQ->execute();
            $stockRes = $stockQ->get_result();

            if ($stockRes && $stockRes->num_rows > 0) {

                $rowStock = $stockRes->fetch_assoc();
                $currentQty = floatval($rowStock['qty']);

                // Deduct
                $newQty = $currentQty - $qty;
                if ($newQty < 0) $newQty = 0;

                // Update stock
                $stockUp = $conn->prepare("
                    UPDATE solid_stock 
                    SET qty=?, updated_at=? 
                    WHERE product_name=?
                ");
                $stockUp->bind_param("dss", $newQty, $device_time, $pname);
                $stockUp->execute();
                $stockUp->close();
            }

            $stockQ->close();
        }

        $solidCheck->close();
    }

    $stmtItem->close();

    /* STEP 4: Transaction Entry */
    $typeForTx = ($payment_type === 'bank') ? 'bank' : 'cash';
    $direction = 'in';
    $category  = 'collection_order';
    $description = "Collection Order #{$order_id}";

    $prevBal = 0.00;
    $balStmt = $conn->prepare("SELECT current_balance FROM transactions WHERE type=? ORDER BY id DESC LIMIT 1");
    $balStmt->bind_param("s", $typeForTx);
    $balStmt->execute();
    $resBal = $balStmt->get_result();

    if ($resBal && $resBal->num_rows > 0) {
        $prevBal = floatval($resBal->fetch_assoc()['current_balance']);
    }
    $balStmt->close();

    $newBal = $prevBal + $final_total;

    $tx = $conn->prepare("
        INSERT INTO transactions 
        (type, category, description, amount, previous_balance, current_balance, direction, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $tx->bind_param("sssdddss", $typeForTx, $category, $description, $final_total, $prevBal, $newBal, $direction, $device_time);
    $tx->execute();
    $tx->close();

    /* STEP 5: Update Balance */
    $balTable = ($typeForTx === 'cash') ? "cash_balance" : "bank_balance";

    $check = $conn->query("SELECT id FROM $balTable WHERE id=1");
    if ($check->num_rows == 0) {
        $conn->query("INSERT INTO $balTable (id, current_balance) VALUES (1, 0)");
    }

    $updateBalance = $conn->prepare("UPDATE $balTable SET current_balance=? WHERE id=1");
    $updateBalance->bind_param("d", $newBal);
    $updateBalance->execute();
    $updateBalance->close();

    // Commit
    $conn->commit();

    echo json_encode([
        "success" => true,
        "message" => "Collection order saved successfully!",
        "order_id" => $order_id,
        "customer_id" => $cust_id,
        "time" => $device_time
    ]);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(["success" => false, "message" => "❌ Error: " . $e->getMessage()]);
}

$conn->close();
?>
