<?php
// ============================
// Q Bistro RawBT Print Receipt (for Dine-In / Remaining Items)
// Optimized for 58mm Thermal Printer
// ============================
require __DIR__ . '/vendor/autoload.php';

use Mike42\Escpos\Printer;
use Mike42\Escpos\PrintConnectors\FilePrintConnector;
use Mike42\Escpos\CapabilityProfile;

// Database connection
include "db.php";
$conn = new mysqli($host, $user, $pass, $db);
if ($conn->connect_error) {
    die("DB connection failed");
}

// Get table number
$table_no = intval($_GET['table_no'] ?? 0);
if ($table_no <= 0) {
    die("❌ Invalid table number");
}

// Fetch items for this table
$sql = "SELECT product_name, qty, price FROM cart_items WHERE table_no=?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $table_no);
$stmt->execute();
$result = $stmt->get_result();

$items = [];
$subtotal = 0;
while ($row = $result->fetch_assoc()) {
    $row['total'] = $row['qty'] * $row['price'];
    $items[] = $row;
    $subtotal += $row['total'];
}
$stmt->close();
$conn->close();

// Totals
$discount = 0;
$tip = 0;
$final_total = $subtotal;

// ==========================================
// Build ESC/POS receipt (optimized for 58mm)
// ==========================================
ob_start();
$profile = CapabilityProfile::load("simple");
$connector = new FilePrintConnector("php://output");
$printer = new Printer($connector, $profile);

// ===== Header =====
$printer->setJustification(Printer::JUSTIFY_CENTER);
$printer->setTextSize(2, 2);
$printer->setEmphasis(true);
$printer->text("Q Bistro\n");
$printer->setEmphasis(false);
$printer->setTextSize(1, 1);
$printer->text("Lucky Market, Beanibazar\n");
$printer->text("Mob: 01341862571\n");
$printer->text(str_repeat("-", 32) . "\n");

// ===== Table Info =====
$printer->setJustification(Printer::JUSTIFY_LEFT);
$printer->setEmphasis(true);
$printer->text("Table: " . $table_no . "\n");
$printer->setEmphasis(false);

// 🔥 Big and bold DINE-IN heading
$printer->setJustification(Printer::JUSTIFY_CENTER);
$printer->setTextSize(2, 2);
$printer->setEmphasis(true);
$printer->text("DINE-IN\n");
$printer->setEmphasis(false);
$printer->setTextSize(1, 1);
$printer->setJustification(Printer::JUSTIFY_LEFT);

$printer->text("Payment: Not Paid\n");
$printer->text("Date: " . date("d-m-Y H:i") . "\n");
$printer->text(str_repeat("-", 32) . "\n");

// ===== Items =====
foreach ($items as $it) {
    $name = substr($it['product_name'], 0, 18);
    $qty = "x" . $it['qty'];
    $price = number_format($it['total'], 2);

    // Big item names
    $printer->setTextSize(1, 1);
    $printer->setEmphasis(true);
    $printer->text($name . "\n");
    $printer->setEmphasis(false);
    $printer->text("   " . str_pad($qty, 6) . str_pad($price, 20, ' ', STR_PAD_LEFT) . "\n");
}
$printer->text(str_repeat("-", 32) . "\n");

// ===== Totals =====
$printer->setJustification(Printer::JUSTIFY_RIGHT);
$printer->setTextSize(1, 1);
$printer->text("Subtotal: " . number_format($subtotal, 2) . "\n");

if ($discount > 0)
    $printer->text("Discount: -" . number_format($discount, 2) . "\n");

if ($tip > 0)
    $printer->text("Tip: " . number_format($tip, 2) . "\n");

// 🔥 Big bold total
$printer->setEmphasis(true);
$printer->setTextSize(2, 2);
$printer->text("TOTAL: " . number_format($final_total, 2) . "\n");
$printer->setEmphasis(false);
$printer->setTextSize(1, 1);
$printer->text(str_repeat("-", 32) . "\n");

// ===== Footer =====
$printer->feed(2);
$printer->setJustification(Printer::JUSTIFY_CENTER);
$printer->setTextSize(1, 1);
$printer->text("Thank you! Visit again\n");
$printer->text(str_repeat("-", 32) . "\n");

// ===== Cut and Close =====
$printer->cut();
$printer->close();

// Capture ESC/POS output
$output = ob_get_clean();
$encodedData = urlencode($output);
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Printing...</title>
<script>
window.onload = function() {
    const escposData = "<?php echo $encodedData; ?>";
    const rawbtUrl = "intent://print?raw=" + escposData + "#Intent;scheme=rawbt;package=ru.a402d.rawbtprinter;end";
    window.location.href = rawbtUrl;
    setTimeout(() => {
        window.location.href = "dashboard.php";
    }, 5000);
};
</script>
<style>
body {
    font-family: monospace;
    text-align: center;
    padding: 50px;
    max-width: 58mm;
    margin: auto;
}
h3 {
    color: #333;
    font-size: 18px;
}
p {
    font-size: 14px;
}
</style>
</head>
<body>
<h3>🖨️ Printing Dine-In Receipt...</h3>
<p>If it doesn’t print automatically, open with RawBT.</p>
</body>
</html>
