// order_collection.js
window.cart = window.cart || [];

function postJSON(url, data) {
  return fetch(url, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(data)
  }).then(async res => {
    const text = await res.text();
    if (!res.ok) throw new Error("Network error: " + res.status + " - " + text);
    try {
      return JSON.parse(text);
    } catch {
      throw new Error("Invalid JSON: " + text);
    }
  });
}

function initCollectionPage() {
  function updateCartTable() {
    const tbody = document.querySelector("#cartTable tbody");
    if (!tbody) return;

    tbody.innerHTML = "";
    let grandTotal = 0;

    window.cart.forEach((item, idx) => {
      const total = (item.price || 0) * (item.qty || 1);
      grandTotal += total;
      const tr = document.createElement("tr");
      tr.innerHTML = `
        <td>${item.name}</td>
        <td><input type="number" min="1" value="${item.qty}" data-index="${idx}" class="qtyInput"></td>
        <td>৳${item.price.toFixed(2)}</td>
        <td>৳${total.toFixed(2)}</td>
        <td><button data-index="${idx}" class="removeBtn">❌</button></td>
      `;
      tbody.appendChild(tr);
    });

    document.getElementById("grandTotal").innerText = grandTotal.toFixed(2);

    // qty change
    document.querySelectorAll(".qtyInput").forEach(input => {
      input.onchange = () => {
        const i = Number(input.dataset.index);
        window.cart[i].qty = Math.max(1, parseInt(input.value) || 1);
        updateCartTable();
      };
    });

    // remove
    document.querySelectorAll(".removeBtn").forEach(btn => {
      btn.onclick = () => {
        const i = Number(btn.dataset.index);
        window.cart.splice(i, 1);
        updateCartTable();
      };
    });
  }

  // add products to cart
  const productContainer = document.querySelector(".products-container");
  if (productContainer) {
    productContainer.addEventListener("click", e => {
      const card = e.target.closest(".product-card");
      if (!card) return;
      const id = parseInt(card.dataset.id);
      const name = card.dataset.name;
      const price = parseFloat(card.dataset.price);
      const exist = window.cart.find(i => i.id === id);
      if (exist) exist.qty += 1;
      else window.cart.push({ id, name, price, qty: 1 });
      updateCartTable();
    });
  }

  // 👉 Handle Submit Collection Order
  const checkoutBtn = document.getElementById("goCheckoutBtn");
  if (checkoutBtn) {
    checkoutBtn.onclick = () => {
      if (window.cart.length === 0) {
        alert("Your cart is empty!");
        return;
      }
      // Save cart to localStorage
      localStorage.setItem("collectionCart", JSON.stringify(window.cart));
      // Redirect to checkout page
      window.location.href = "collection_checkout.php";
    };
  }

  updateCartTable();
}
