// order.js (updated clean version)
window.cart = window.cart || [];

function postJSON(url, data) {
  return fetch(url, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(data)
  }).then(async res => {
    const text = await res.text();
    if (!res.ok) throw new Error("Network error: " + res.status + " - " + text);
    try {
      return JSON.parse(text);
    } catch {
      throw new Error("Invalid JSON from " + url + ": " + text);
    }
  });
}

function initOrderPage(table = 0, type = "dinein") {
  if (!(type === "dinein" && table > 0)) window.cart = [];

  let activeOrderId = 0;
  let isBooked = false;

  function updateCartTable() {
    const tbody = document.querySelector("#cartTable tbody");
    if (!tbody) return;

    tbody.innerHTML = "";
    let grandTotal = 0;

    window.cart.forEach((item, index) => {
      const total = (item.price || 0) * (item.qty || 1);
      grandTotal += total;

      const tr = document.createElement("tr");
      tr.innerHTML = `
        <td>${item.name}</td>
        <td><input type="number" min="1" value="${item.qty}" data-index="${index}" class="qtyInput"></td>
        <td>৳${(item.price || 0).toFixed(2)}</td>
        <td>৳${total.toFixed(2)}</td>
        <td> <button data-index="${index}" class="removeBtn">❌</button> </td>
      `;
      tbody.appendChild(tr);
    });

    const grandTotalElem = document.getElementById("grandTotal");
    if (grandTotalElem) grandTotalElem.innerText = grandTotal.toFixed(2);

    // qty change
    document.querySelectorAll(".qtyInput").forEach(input => {
      input.onchange = () => {
        const idx = Number(input.dataset.index);
        window.cart[idx].qty = Math.max(1, parseInt(input.value) || 1);
        updateCartTable();
      };
    });

    // remove
    document.querySelectorAll(".removeBtn").forEach(btn => {
      btn.onclick = () => {
        const idx = Number(btn.dataset.index);
        window.cart.splice(idx, 1);
        updateCartTable();
      };
    });
  }

  // --- Load existing order for dine-in ---
  if (type === "dinein" && table > 0) {
    fetch(`get_order.php?table=${table}`)
      .then(res => {
        if (!res.ok) throw new Error("Failed to fetch order data");
        return res.json();
      })
      .then(data => {
        if (data && data.status === "success" && Array.isArray(data.cart)) {
          window.cart = data.cart;
          activeOrderId = data.order_id || 0;
          isBooked = !!activeOrderId;
        } else {
          window.cart = [];
          isBooked = false;
          activeOrderId = 0;
        }
        setupButtons();
        updateCartTable();
      })
      .catch(err => {
        console.error("Failed to load order:", err);
        setupButtons();
        updateCartTable();
      });
  } else {
    setupButtons();
    updateCartTable();
  }

  // --- Add product to cart ---
  const productContainer = document.querySelector(".products-container") || document;
  productContainer.addEventListener("click", e => {
    const card = e.target.closest(".product-card");
    if (!card) return;
    const id = parseInt(card.dataset.id);
    const name = card.dataset.name;
    const price = parseFloat(card.dataset.price);
    const exist = window.cart.find(i => i.id === id);
    if (exist) exist.qty += 1;
    else window.cart.push({ id, name, price, qty: 1 });
    updateCartTable();
  });

  // --- Buttons ---
  function setupButtons() {
    const footer = document.querySelector(".cart-footer");
    if (!footer) return;

    // remove old container
    let old = document.getElementById("__actionBtns");
    if (old) old.remove();

    // container
    const box = document.createElement("div");
    box.id = "__actionBtns";
    box.style.display = "flex";
    box.style.gap = "12px";
    box.style.marginTop = "10px";

    if (isBooked) {
      // Update button
      const updateBtn = document.createElement("button");
      updateBtn.innerText = "Update Order";
      updateBtn.onclick = () => {
        if (!window.cart.length) { alert("Cart is empty!"); return; }
        updateBtn.disabled = true;
        updateBtn.innerText = "Updating...";
        postJSON("update_order.php", { table, type, cart: window.cart })
          .then(resp => {
            if (resp.status === "success") {
              alert("Order updated successfully");
              if (resp.receipt_url) window.open(resp.receipt_url, "_blank");
              setTimeout(() => loadPage ? loadPage('dinein.php') : location.reload(), 900);
            } else throw new Error(resp.message || "Unknown error");
          })
          .catch(err => alert("Update failed: " + err.message))
          .finally(() => {
            updateBtn.disabled = false;
            updateBtn.innerText = "Update Order";
          });
      };
      box.appendChild(updateBtn);
    } else {
      // Send button
      const sendBtn = document.createElement("button");
      sendBtn.innerText = "Send Order";
      sendBtn.onclick = () => {
        if (!window.cart.length) { alert("Cart is empty!"); return; }
        sendBtn.disabled = true;
        sendBtn.innerText = "Sending...";
        postJSON("submit_order.php", { table, type, cart: window.cart })
          .then(resp => {
            if (resp.status === "success") {
              alert("Order sent successfully");
              if (resp.receipt_url) window.open(resp.receipt_url, "_blank");
              setTimeout(() => loadPage ? loadPage('dinein.php') : location.reload(), 900);
            } else throw new Error(resp.message || "Unknown error");
          })
          .catch(err => alert("Send failed: " + err.message))
          .finally(() => {
            sendBtn.disabled = false;
            sendBtn.innerText = "Send Order";
          });
      };
      box.appendChild(sendBtn);
    }

    // Checkout button (always last)
    const checkoutBtn = document.createElement("button");
    checkoutBtn.innerText = "Checkout";
    checkoutBtn.onclick = () => {
      if (!window.cart.length) { alert("Cart is empty!"); return; }
      sessionStorage.setItem("currentCart", JSON.stringify(window.cart));
      sessionStorage.setItem("currentTable", table);
      sessionStorage.setItem("orderType", type);
      location.href = `checkout.php?table=${table}&type=${type}`;
    };
    box.appendChild(checkoutBtn);

    footer.appendChild(box);
  }
}

