<?php
include "db.php";
header("Content-Type: application/json");

// Read JSON input
$input = file_get_contents("php://input");
$data = json_decode($input, true);

// Validate input
if (!isset($data["id"]) || !isset($data["qty"])) {
    echo json_encode(["success" => false, "message" => "Invalid input"]);
    exit;
}

$id = intval($data["id"]);
$fry_qty = floatval($data["qty"]);

if ($id <= 0 || $fry_qty <= 0) {
    echo json_encode(["success" => false, "message" => "Invalid product ID or quantity"]);
    exit;
}

// ✅ Fetch product info (including qty_type)
$q = $conn->prepare("SELECT product_name, qty, qty_type FROM current_stock WHERE id = ?");
$q->bind_param("i", $id);
$q->execute();
$res = $q->get_result();

if ($res->num_rows === 0) {
    echo json_encode(["success" => false, "message" => "Product not found"]);
    exit;
}

$product = $res->fetch_assoc();
$q->close();

// ✅ Check if enough stock
if ($product["qty"] < $fry_qty) {
    echo json_encode(["success" => false, "message" => "Not enough stock"]);
    exit;
}

$new_qty = $product["qty"] - $fry_qty;

// ✅ Start transaction
$conn->begin_transaction();

try {
    // Update current stock
    $update = $conn->prepare("UPDATE current_stock SET qty = ?, updated_at = NOW() WHERE id = ?");
    $update->bind_param("di", $new_qty, $id);
    $update->execute();
    $update->close();

    // ✅ Insert fry record including qty_type
    $insert = $conn->prepare("INSERT INTO fry_item (product_name, qty, qty_type, created_at) VALUES (?, ?, ?, NOW())");
    $insert->bind_param("sds", $product["product_name"], $fry_qty, $product["qty_type"]);
    $insert->execute();
    $insert->close();

    $conn->commit();
    echo json_encode(["success" => true, "message" => "✅ Fry item saved successfully"]);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(["success" => false, "message" => "❌ Transaction failed: " . $e->getMessage()]);
}

$conn->close();
?>
