<?php
header('Content-Type: application/json');
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . "/php_errors.log");

include "db.php";

$conn = new mysqli($host, $user, $pass, $db);
if ($conn->connect_error) {
  echo json_encode(["status" => "error", "message" => "DB connection failed"]);
  exit;
}

$input = file_get_contents("php://input");
$data  = json_decode($input, true);
if (!$data) $data = $_POST;

if (empty($data)) {
  echo json_encode(["status" => "error", "message" => "No input data"]);
  exit;
}

/* Time from device or fallback */
if (!empty($data['device_time'])) {
  $device_time = date("Y-m-d H:i:s", strtotime($data['device_time']));
} else {
  date_default_timezone_set("Asia/Dhaka");
  $device_time = date("Y-m-d H:i:s");
}

/* Variables */
$table            = intval($data['table'] ?? 0);
$type             = $data['type'] ?? 'dinein';
$grand_total      = floatval($data['grand_total'] ?? 0);
$customer_name    = trim($data['customer_name'] ?? 'Guest');
$customer_phone   = trim($data['customer_phone'] ?? $data['customer_mobile'] ?? '');
$customer_address = trim($data['customer_address'] ?? '');
$discount         = floatval($data['discount'] ?? 0);
$tip              = floatval($data['tip'] ?? 0);
$payment_type     = strtolower($data['payment_type'] ?? 'cash');
$final_total      = max(0, $grand_total - $discount + $tip);

try {
  $conn->begin_transaction();

  /* Insert or update customer */
  if (!empty($customer_phone)) {
    $stmtCheck = $conn->prepare("SELECT id FROM customers WHERE mobile=?");
    $stmtCheck->bind_param("s", $customer_phone);
    $stmtCheck->execute();
    $res = $stmtCheck->get_result();

    if ($res && $res->num_rows > 0) {
      $row = $res->fetch_assoc();
      $cust_id = $row['id'];

      $stmtUpdCust = $conn->prepare("UPDATE customers SET name=?, address=? WHERE id=?");
      $stmtUpdCust->bind_param("ssi", $customer_name, $customer_address, $cust_id);
      $stmtUpdCust->execute();
      $stmtUpdCust->close();

    } else {
      $stmtCust = $conn->prepare("INSERT INTO customers (name, mobile, address, created_at) VALUES (?, ?, ?, ?)");
      $stmtCust->bind_param("ssss", $customer_name, $customer_phone, $customer_address, $device_time);
      $stmtCust->execute();
      $cust_id = $stmtCust->insert_id;
      $stmtCust->close();
    }
    $stmtCheck->close();
  }

  /* Insert order */
  $stmt = $conn->prepare("INSERT INTO orders (order_type, customer_name, customer_phone, table_no, payment_type, subtotal, discount, tip, final_total, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
  $stmt->bind_param("sssisdidds", $type, $customer_name, $customer_phone, $table, $payment_type, $grand_total, $discount, $tip, $final_total, $device_time);
  $stmt->execute();
  $order_id = $stmt->insert_id;
  $stmt->close();

  if ($order_id <= 0) throw new Exception("Order insert failed");

  /* Fetch cart items */
  $stmtCart = $conn->prepare("SELECT * FROM cart_items WHERE table_no=?");
  $stmtCart->bind_param("i", $table);
  $stmtCart->execute();
  $cartRes = $stmtCart->get_result();

  if ($cartRes->num_rows === 0) throw new Exception("Cart empty for this table");

  /* Insert order items */
  $stmtItems = $conn->prepare("INSERT INTO order_items (order_id, product_id, product_name, qty, price, total) VALUES (?, ?, ?, ?, ?, ?)");

  while ($row = $cartRes->fetch_assoc()) {

    $pid   = intval($row['product_id']);
    $pname = $row['product_name'];
    $qty   = floatval($row['qty']);
    $price = floatval($row['price']);
    $total = floatval($row['total']);

    /* Insert order item */
    $stmtItems->bind_param("iisidd", $order_id, $pid, $pname, $qty, $price, $total);
    if (!$stmtItems->execute()) {
      throw new Exception("Insert failed (order_items): " . $stmtItems->error);
    }

    /*---------------------------------------------
      SOLID PRODUCT CHECK + STOCK DEDUCTION
    ----------------------------------------------*/

    /* Check if product is Solid Product */
    $checkPU = $conn->prepare("
      SELECT id FROM pu_product
      WHERE p_product_name = ?
        AND product_type = 'Solid Product'
      LIMIT 1
    ");
    $checkPU->bind_param("s", $pname);
    $checkPU->execute();
    $puRes = $checkPU->get_result();
    $checkPU->close();

    if ($puRes->num_rows > 0) {

      /* Fetch solid stock */
      $checkSolid = $conn->prepare("
        SELECT qty FROM solid_stock
        WHERE product_name = ?
        LIMIT 1
      ");
      $checkSolid->bind_param("s", $pname);
      $checkSolid->execute();
      $solidRes = $checkSolid->get_result();
      $checkSolid->close();

      if ($solidRes->num_rows > 0) {

        $solid = $solidRes->fetch_assoc();
        $oldQty = floatval($solid['qty']);

        /* Deduct */
        $newQty = $oldQty - $qty;
        if ($newQty < 0) $newQty = 0;

        /* Update stock */
        $updSolid = $conn->prepare("
          UPDATE solid_stock
          SET qty = ?, updated_at = NOW()
          WHERE product_name = ?
        ");
        $updSolid->bind_param("ds", $newQty, $pname);
        $updSolid->execute();
        $updSolid->close();
      }
    }
  }  // <-- END WHILE LOOP ✔

  $stmtItems->close();
  $stmtCart->close();

  /* Clear cart */
  $stmtDel = $conn->prepare("DELETE FROM cart_items WHERE table_no=?");
  $stmtDel->bind_param("i", $table);
  $stmtDel->execute();
  $stmtDel->close();

  /* Release table */
  if ($type === "dinein" && $table > 0) {
    $stmtTbl = $conn->prepare("UPDATE dine_in_tables SET status='available' WHERE id=?");
    $stmtTbl->bind_param("i", $table);
    $stmtTbl->execute();
    $stmtTbl->close();
  }

  /* Transactions */
  $typeForTx = ($payment_type === 'bank') ? 'bank' : 'cash';
  $direction = 'in';
  $category  = 'order';
  $description = "Order #{$order_id} (Table {$table})";

  $prevBal = 0.00;
  $balStmt = $conn->prepare("SELECT current_balance FROM transactions WHERE type=? ORDER BY id DESC LIMIT 1");
  $balStmt->bind_param("s", $typeForTx);
  $balStmt->execute();
  $resBal = $balStmt->get_result();
  if ($resBal && $resBal->num_rows > 0) {
    $prevBal = floatval($resBal->fetch_assoc()['current_balance']);
  }
  $balStmt->close();

  $newBal = $prevBal + $final_total;

  $tx = $conn->prepare("INSERT INTO transactions (type, category, description, amount, previous_balance, current_balance, direction, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
  $tx->bind_param("sssdddss", $typeForTx, $category, $description, $final_total, $prevBal, $newBal, $direction, $device_time);
  $tx->execute();
  $tx->close();

  if ($typeForTx === 'cash') {
    $stmtUpd = $conn->prepare("UPDATE cash_balance SET current_balance=? WHERE id=1");
  } else {
    $stmtUpd = $conn->prepare("UPDATE bank_balance SET current_balance=? WHERE id=1");
  }
  $stmtUpd->bind_param("d", $newBal);
  $stmtUpd->execute();
  $stmtUpd->close();

  $conn->commit();

  echo json_encode([
    "status" => "success",
    "message" => "Checkout completed successfully",
    "order_id" => $order_id,
    "device_time" => $device_time,
    "receipt_url" => "bill_receipt.php?order_id=" . $order_id
  ]);

} catch (Exception $e) {

  $conn->rollback();
  echo json_encode([
    "status" => "error",
    "message" => "❌ Checkout failed: " . $e->getMessage()
  ]);
}

$conn->close();
?>
