<?php
include "db.php";
header('Content-Type: application/json');

if ($conn->connect_error) {
    echo json_encode(["success" => false, "message" => "Database connection failed"]);
    exit;
}

error_reporting(E_ALL);
ini_set('display_errors', 1);

// ✅ Get POST data
$name       = $_POST['expense_name'] ?? '';
$amount     = floatval($_POST['amount'] ?? 0);
$notes      = $_POST['notes'] ?? '';
$payment    = strtolower(trim($_POST['payment_type'] ?? 'cash')); // cash / bank / card

if (empty($name) || $amount <= 0) {
    echo json_encode(["success" => false, "message" => "Invalid expense data"]);
    exit;
}

try {
    $conn->begin_transaction();

    // ✅ 1️⃣ Insert into expenses
    $stmt = $conn->prepare("
        INSERT INTO expenses (expense_name, amount, notes, payment_type, created_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    if (!$stmt) throw new Exception("Prepare failed (expenses): " . $conn->error);
    $stmt->bind_param("sdss", $name, $amount, $notes, $payment);
    $stmt->execute();
    $expense_id = $stmt->insert_id;
    $stmt->close();

    // ✅ 2️⃣ Determine type (bank for bank/card)
    $typeForTx = in_array($payment, ['bank', 'card']) ? 'bank' : 'cash';

    $direction   = 'out';
    $category    = 'expense';
    $description = "Expense #{$expense_id} - {$name}";

    // ✅ 3️⃣ Previous balance
    $prevBal = 0.00;
    $balStmt = $conn->prepare("SELECT current_balance FROM transactions WHERE type=? ORDER BY id DESC LIMIT 1");
    if ($balStmt) {
        $balStmt->bind_param("s", $typeForTx);
        $balStmt->execute();
        $res = $balStmt->get_result();
        if ($res && $res->num_rows > 0) {
            $prevBal = floatval($res->fetch_assoc()['current_balance']);
        }
        $balStmt->close();
    }

    // ✅ 4️⃣ New balance (money out)
    $newBal = $prevBal - $amount;

    // ✅ 5️⃣ Insert transaction
    $tx = $conn->prepare("
        INSERT INTO transactions 
        (type, category, description, amount, previous_balance, current_balance, direction, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    if (!$tx) throw new Exception("Prepare failed (transactions): " . $conn->error);
    $tx->bind_param("sssddds", $typeForTx, $category, $description, $amount, $prevBal, $newBal, $direction);
    $tx->execute();
    $tx->close();

    // ✅ 6️⃣ Update balance table
    $balTable = ($typeForTx === 'cash') ? "cash_balance" : "bank_balance";

    // Ensure exists
    $check = $conn->query("SELECT id FROM $balTable WHERE id=1");
    if ($check->num_rows == 0) {
        $conn->query("INSERT INTO $balTable (id, current_balance) VALUES (1, 0)");
    }

    $updateBal = $conn->prepare("UPDATE $balTable SET current_balance = ? WHERE id = 1");
    $updateBal->bind_param("d", $newBal);
    $updateBal->execute();
    $updateBal->close();

    // ✅ Commit
    $conn->commit();

    echo json_encode([
        "success" => true,
        "message" => "Expense added successfully",
        "expense_id" => $expense_id,
        "type" => $typeForTx,
        "previous_balance" => $prevBal,
        "current_balance" => $newBal
    ]);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(["success" => false, "message" => "❌ Error: " . $e->getMessage()]);
}

$conn->close();
?>
