<?php
session_start();
include "db.php";

if (!isset($_SESSION['username'])) {
    die("Please login first.");
}

$username = $_SESSION['username'];
$action   = $_POST['action'] ?? '';
$amount   = floatval($_POST['amount'] ?? 0);
$notes    = trim($_POST['notes'] ?? '');
$bank     = trim($_POST['bank_name'] ?? '');

if ($amount <= 0) {
    die("Invalid amount.");
}

/* 🔹 Ensure cb_balance table exists */
$conn->query("CREATE TABLE IF NOT EXISTS cb_balance (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(100) NOT NULL,
    action_type ENUM('add_cash','cash_to_bank','withdraw') NOT NULL,
    amount DECIMAL(12,2) NOT NULL,
    notes TEXT,
    bank_name VARCHAR(150),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
)");

/* 🔹 Ensure one record exists in cash/bank balance */
$conn->query("INSERT IGNORE INTO cash_balance (id, current_balance, updated_at) VALUES (1, 0, NOW())");
$conn->query("INSERT IGNORE INTO bank_balance (id, current_balance, updated_at) VALUES (1, 0, NOW())");

/* 🔹 Fetch current balances safely */
$cashRes = $conn->query("SELECT current_balance FROM cash_balance WHERE id=1");
$cashBal = ($cashRes && $cashRes->num_rows > 0) ? floatval($cashRes->fetch_assoc()['current_balance']) : 0;

$bankRes = $conn->query("SELECT current_balance FROM bank_balance WHERE id=1");
$bankBal = ($bankRes && $bankRes->num_rows > 0) ? floatval($bankRes->fetch_assoc()['current_balance']) : 0;

/* 🔹 Record in cb_balance table */
$stmt = $conn->prepare("INSERT INTO cb_balance (username, action_type, amount, notes, bank_name) VALUES (?, ?, ?, ?, ?)");
$stmt->bind_param("ssdss", $username, $action, $amount, $notes, $bank);
$stmt->execute();
$stmt->close();

/* 🔹 Process each action properly */
switch ($action) {
    case 'add_cash':
        $prevBal = $cashBal;
        $cashBal += $amount;
        $conn->query("UPDATE cash_balance SET current_balance=$cashBal, updated_at=NOW() WHERE id=1");

        // record transaction
        $desc = "Cash added manually by $username";
        $stmt = $conn->prepare("INSERT INTO transactions (type, category, reference_id, description, amount, previous_balance, current_balance, direction, created_at)
                                VALUES ('in', 'Add Cash', 0, ?, ?, ?, ?, 'in', NOW())");
        $stmt->bind_param("sddd", $desc, $amount, $prevBal, $cashBal);
        $stmt->execute();
        $stmt->close();

        echo "✅ Cash added successfully!";
        break;

    case 'cash_to_bank':
        if ($cashBal < $amount) {
            die("❌ Not enough cash balance.");
        }

        $prevCash = $cashBal;
        $prevBank = $bankBal;

        // update balances
        $cashBal -= $amount;
        $bankBal += $amount;

        $conn->query("UPDATE cash_balance SET current_balance=$cashBal, updated_at=NOW() WHERE id=1");
        $conn->query("UPDATE bank_balance SET current_balance=$bankBal, updated_at=NOW() WHERE id=1");

        // record transaction (cash out)
        $desc = "Cash to bank transfer ($bank) by $username";
        $stmt = $conn->prepare("INSERT INTO transactions (type, category, reference_id, description, amount, previous_balance, current_balance, direction, created_at)
                                VALUES ('out', 'Cash to Bank', 0, ?, ?, ?, ?, 'out', NOW())");
        $stmt->bind_param("sddd", $desc, $amount, $prevCash, $cashBal);
        $stmt->execute();
        $stmt->close();

        // record transaction (bank in)
        $desc = "Cash received in bank ($bank) from cash transfer by $username";
        $stmt = $conn->prepare("INSERT INTO transactions (type, category, reference_id, description, amount, previous_balance, current_balance, direction, created_at)
                                VALUES ('in', 'Bank Deposit', 0, ?, ?, ?, ?, 'in', NOW())");
        $stmt->bind_param("sddd", $desc, $amount, $prevBank, $bankBal);
        $stmt->execute();
        $stmt->close();

        echo "🏦 Cash transferred to bank successfully!";
        break;

    case 'withdraw':
        if ($cashBal < $amount) {
            die("❌ Not enough cash balance.");
        }

        $prevBal = $cashBal;
        $cashBal -= $amount;
        $conn->query("UPDATE cash_balance SET current_balance=$cashBal, updated_at=NOW() WHERE id=1");

        $desc = "Cash withdrawn by $username";
        $stmt = $conn->prepare("INSERT INTO transactions (type, category, reference_id, description, amount, previous_balance, current_balance, direction, created_at)
                                VALUES ('out', 'Cash Withdraw', 0, ?, ?, ?, ?, 'out', NOW())");
        $stmt->bind_param("sddd", $desc, $amount, $prevBal, $cashBal);
        $stmt->execute();
        $stmt->close();

        echo "💸 Cash withdrawn successfully!";
        break;

    default:
        echo "❌ Unknown action.";
        break;
}

$conn->close();
?>
