<?php
// ============================
// Q Bistro RawBT Print Receipt
// Optimized for 58mm thermal printer
// ============================

require __DIR__ . '/vendor/autoload.php';

use Mike42\Escpos\Printer;
use Mike42\Escpos\PrintConnectors\FilePrintConnector;
use Mike42\Escpos\CapabilityProfile;

// Database connect
include "db.php";
$conn = new mysqli($host, $user, $pass, $db);
if ($conn->connect_error) {
    die("DB connection failed");
}

// Get order ID
$order_id = intval($_GET['order_id'] ?? 0);
if ($order_id <= 0) {
    die("❌ Invalid order id.");
}

// Fetch order
$sql = "SELECT * FROM orders WHERE id=?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $order_id);
$stmt->execute();
$order = $stmt->get_result()->fetch_assoc();
$stmt->close();
if (!$order) {
    die("❌ Order not found.");
}

// Fetch items
$sqlItems = "SELECT product_name, qty, price, total FROM order_items WHERE order_id=?";
$stmtItems = $conn->prepare($sqlItems);
$stmtItems->bind_param("i", $order_id);
$stmtItems->execute();
$items = $stmtItems->get_result()->fetch_all(MYSQLI_ASSOC);
$stmtItems->close();
$conn->close();

// Build ESC/POS receipt
ob_start();
$profile = CapabilityProfile::load("simple");
$connector = new FilePrintConnector("php://output");
$printer = new Printer($connector, $profile);

// ======= Header =======
$printer->setJustification(Printer::JUSTIFY_CENTER);
$printer->setTextSize(2, 2); // Bigger title
$printer->setEmphasis(true);
$printer->text("Q Bistro\n");
$printer->setEmphasis(false);
$printer->setTextSize(1, 1);
$printer->text("Lucky Market, Beanibazar\n");
$printer->text("Mob: 01341862571\n");
$printer->text(str_repeat("-", 32) . "\n");

// ======= Order Info =======
$printer->setJustification(Printer::JUSTIFY_LEFT);
$printer->setEmphasis(true);
$printer->text("Order No: " . $order['id'] . "\n");
$printer->setEmphasis(false);
$printer->text("Table: " . $order['table_no'] . "\n");

// 🔥 Make order type big and bold (e.g. Dine-In, Takeaway, Collection)
$printer->setEmphasis(true);
$printer->setTextSize(2, 1);
$printer->text("Type: " . strtoupper($order['order_type']) . "\n");
$printer->setTextSize(1, 1);
$printer->setEmphasis(false);

if (!empty($order['customer_name'])) $printer->text("Customer: " . $order['customer_name'] . "\n");
if (!empty($order['customer_phone'])) $printer->text("Phone: " . $order['customer_phone'] . "\n");
$printer->text("Payment: " . ucfirst($order['payment_type']) . "\n");
$printer->text("Date: " . date("d-m-Y H:i", strtotime($order['created_at'])) . "\n");
$printer->text(str_repeat("-", 32) . "\n");

// ======= Items =======
foreach ($items as $it) {
    $name = substr($it['product_name'], 0, 18);
    $qty = "x" . $it['qty'];
    $price = number_format($it['total'], 2);

    // Larger font for item names
    $printer->setTextSize(1, 1);
    $printer->setEmphasis(true);
    $printer->text($name . "\n");
    $printer->setEmphasis(false);
    $printer->text("   " . str_pad($qty, 6) . str_pad($price, 20, ' ', STR_PAD_LEFT) . "\n");
}
$printer->text(str_repeat("-", 32) . "\n");

// ======= Totals =======
$printer->setJustification(Printer::JUSTIFY_RIGHT);
$printer->setTextSize(1, 1);
$printer->text("Subtotal: " . number_format($order['subtotal'], 2) . "\n");
if ($order['discount'] > 0) $printer->text("Discount: -" . number_format($order['discount'], 2) . "\n");
if ($order['tip'] > 0) $printer->text("Tip: " . number_format($order['tip'], 2) . "\n");

// 🔥 Big, bold TOTAL
$printer->setTextSize(2, 2);
$printer->setEmphasis(true);
$printer->text("TOTAL: " . number_format($order['final_total'], 2) . "\n");
$printer->setEmphasis(false);
$printer->setTextSize(1, 1);
$printer->text(str_repeat("-", 32) . "\n");

// ======= Footer =======
$printer->feed(2);
$printer->setJustification(Printer::JUSTIFY_CENTER);
$printer->setTextSize(1, 1);
$printer->text("Thank you! Visit again\n");
$printer->text(str_repeat("-", 32) . "\n");

// ======= Finish =======
$printer->cut();
$printer->close();

// Capture output for RawBT
$output = ob_get_clean();
$encodedData = urlencode($output);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Printing...</title>
    <script>
        window.onload = function() {
            const escposData = "<?php echo $encodedData; ?>";
            const rawbtUrl = "intent://print?raw=" + escposData + "#Intent;scheme=rawbt;package=ru.a402d.rawbtprinter;end";
            window.location.href = rawbtUrl;

            // Redirect back to dashboard after 5 seconds
            setTimeout(() => {
                window.location.href = "dashboard.php";
            }, 5000);
        };
    </script>
    <style>
        body {
            font-family: monospace;
            text-align: center;
            padding: 50px;
            max-width: 58mm;
            margin: auto;
        }
        h3 {
            color: #333;
            font-size: 18px;
        }
        p {
            font-size: 14px;
        }
    </style>
</head>
<body>
    <h3>🖨️ Printing Receipt...</h3>
    <p>If it doesn’t print automatically, open with RawBT.</p>
</body>
</html>
